/*
 * Decompiled with CFR 0.152.
 */
package inform.agent.files;

import inform.adt.IO;
import inform.adt.taggedio.ByteArrayOutputStream;
import inform.agent.Core;
import inform.agent.files.BFS;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;

public class FileCache {
    public static final String CACHE_EXTENTION = ".cache";
    public final File rootFolder;
    private boolean initialized = false;
    private final int hashSize;
    private long maxCacheSize = 0L;
    private long minFileCount = 0L;
    private final HashMap<String, CacheItem> hashMap = new HashMap();
    private final CacheItemComparator fileComparator = new CacheItemComparator();
    private CacheItem head = null;
    private CacheItem tail = null;
    private long cacheSize = 0L;
    private int fileCount = 0;

    public FileCache(File rootFolder, int maxFileCount, long maxCacheSize) {
        this.rootFolder = rootFolder;
        this.hashSize = maxFileCount;
        this.maxCacheSize = maxCacheSize;
        this.minFileCount = maxFileCount / 100;
        if (this.minFileCount == 0L) {
            this.minFileCount = 10L;
        }
    }

    public void put(String key, byte[] content, int from, int length) throws IOException {
        if (length == 0) {
            this.putEmpty(key);
        } else {
            this.putFile(key, content, from, length);
        }
    }

    public void put(String key, byte[] content, int length) throws IOException {
        this.put(key, content, 0, length);
    }

    public void put(String key, byte[] content) throws IOException {
        if (content == null) {
            this.putEmpty(key);
        } else {
            this.putFile(key, content, 0, content.length);
        }
    }

    public void put(String key, ByteArrayOutputStream content) throws IOException {
        if (content == null) {
            this.putEmpty(key);
        } else {
            this.putFile(key, content.internalBuffer(), 0, content.size());
        }
    }

    public byte[] get(String key, long time) throws IOException {
        return this.getFile(key, time);
    }

    public void remove(String key) {
        this.removeFile(key);
    }

    public int getHashMapSize() {
        return this.fileCount;
    }

    private void putEmpty(String key) {
        this.removeFile(key);
    }

    private File getCacheFile(String key) {
        return new File(this.rootFolder, key + CACHE_EXTENTION);
    }

    private synchronized boolean needResampling(long cacheSize) {
        if ((long)this.fileCount > this.minFileCount && this.maxCacheSize > 0L && this.maxCacheSize < cacheSize) {
            return true;
        }
        return this.hashSize < this.fileCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void putFile(String key, byte[] content, int from, int length) throws IOException {
        if (!this.initialized) {
            this.loadFromFileSystem();
        }
        while (this.tail != null && this.needResampling(this.cacheSize + (long)length)) {
            this.cacheSize -= this.tail.size;
            BFS.deleteFileSilent(this.tail.file, null, false);
            this.hashMap.remove(key);
            this.removeItem(this.tail);
        }
        CacheItem item = this.hashMap.get(key);
        if (item == null) {
            item = new CacheItem(key, this.getCacheFile(key));
            this.hashMap.put(key, item);
            ++this.fileCount;
        } else {
            this.cacheSize -= item.size;
        }
        try (FileOutputStream out = new FileOutputStream(item.file);){
            out.write(content, from, length);
            out.flush();
        }
        item.time = System.currentTimeMillis();
        item.size = length;
        this.cacheSize += (long)length;
        if (!item.file.setLastModified(item.time)) {
            Core.logger.info("\u041d\u0435 \u0432\u043e\u0437\u043c\u043e\u0436\u043d\u043e \u0438\u0437\u043c\u0435\u043d\u0438\u0442\u044c \u0432\u0440\u0435\u043c\u044f \u0434\u043e\u0441\u0442\u0443\u043f\u0430 \u043a \u0444\u0430\u0439\u043b\u0443 '" + item.file.getAbsolutePath() + "'");
        }
        this.moveToTop(item);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized byte[] getFile(String key, long time) throws IOException {
        byte[] byArray;
        CacheItem item;
        if (!this.initialized) {
            this.loadFromFileSystem();
        }
        if ((item = this.hashMap.get(key)) == null) {
            File file = this.getCacheFile(key);
            if (file == null) {
                return null;
            }
            item = new CacheItem(key, file);
            item.size = item.file.length();
            item.time = item.file.lastModified();
            this.hashMap.put(key, item);
        }
        if (item.time < time) {
            BFS.deleteFileSilent(item.file, null, false);
            this.hashMap.remove(key);
            this.removeItem(item);
            this.cacheSize -= item.size;
            return null;
        }
        FileInputStream in = new FileInputStream(item.file);
        try {
            ByteArrayOutputStream content = new ByteArrayOutputStream();
            IO.transfer(in, content);
            item.time = System.currentTimeMillis();
            if (!item.file.setLastModified(item.time)) {
                Core.logger.info("\u041d\u0435 \u0432\u043e\u0437\u043c\u043e\u0436\u043d\u043e \u0438\u0437\u043c\u0435\u043d\u0438\u0442\u044c \u0432\u0440\u0435\u043c\u044f \u0434\u043e\u0441\u0442\u0443\u043f\u0430 \u043a \u0444\u0430\u0439\u043b\u0443 '" + item.file.getAbsolutePath() + "'");
            }
            this.moveToTop(item);
            this.cacheSize -= item.size;
            this.cacheSize += (long)content.size();
            byArray = content.toByteArray();
        }
        catch (Throwable throwable) {
            try {
                in.close();
                throw throwable;
            }
            catch (FileNotFoundException e) {
                return null;
            }
        }
        in.close();
        return byArray;
    }

    private void removeFile(String key) {
        if (!this.initialized) {
            this.loadFromFileSystem();
        }
        File file = null;
        CacheItem item = this.hashMap.get(key);
        if (item == null) {
            file = this.getCacheFile(key);
        } else {
            file = item.file;
            this.hashMap.remove(key);
            this.removeItem(item);
            this.cacheSize -= item.size;
        }
        BFS.deleteFileSilent(file, null, false);
    }

    public synchronized void loadFromFileSystem() {
        this.initialized = true;
        this.fileCount = 0;
        this.hashMap.clear();
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(FileCache.CACHE_EXTENTION);
            }
        };
        File[] loadedList = this.rootFolder.listFiles(filter);
        this.cacheSize = 0L;
        if (loadedList == null || loadedList.length == 0) {
            return;
        }
        FileItem[] fileList = new FileItem[loadedList.length];
        int index = 0;
        for (File file : loadedList) {
            fileList[index++] = new FileItem(file);
        }
        Arrays.sort(fileList, this.fileComparator);
        for (FileItem fileItem : fileList) {
            if (!fileItem.isFile) continue;
            if (this.needResampling(this.cacheSize)) {
                BFS.deleteFileSilent(fileItem.file, null, false);
                continue;
            }
            String name = fileItem.name;
            CacheItem item = new CacheItem(name.substring(0, name.length() - CACHE_EXTENTION.length()), fileItem.file);
            item.size = fileItem.size;
            item.time = fileItem.time;
            this.hashMap.put(item.key, item);
            this.pushBackItem(item);
            this.cacheSize += item.size;
        }
    }

    private void removeItem(CacheItem item) {
        assert (item != null);
        if (this.head == item) {
            this.head = item.next;
        }
        if (this.tail == item) {
            this.tail = item.prev;
        }
        if (item.prev != null) {
            item.prev.next = item.next;
        }
        if (item.next != null) {
            item.next.prev = item.prev;
        }
        item.prev = null;
        item.next = null;
        --this.fileCount;
    }

    private void moveToTop(CacheItem item) {
        this.removeItem(item);
        if (this.head != null) {
            item.next = this.head;
            this.head.prev = item;
            this.head = item;
        } else {
            this.head = this.tail = item;
        }
        ++this.fileCount;
    }

    private void moveToBottom(CacheItem item) {
        this.removeItem(item);
        if (this.tail != null) {
            item.prev = this.tail;
            this.tail.next = item;
            this.tail = item;
        } else {
            this.head = this.tail = item;
        }
        ++this.fileCount;
    }

    private void pushFrontItem(CacheItem item) {
        assert (item != null);
        assert (item.prev == null);
        assert (item.next == null);
        if (this.head != null) {
            item.next = this.head;
            this.head.prev = item;
            this.head = item;
        } else {
            this.head = this.tail = item;
        }
        ++this.fileCount;
    }

    private void pushBackItem(CacheItem item) {
        assert (item != null);
        assert (item.prev == null);
        assert (item.next == null);
        if (this.tail != null) {
            item.prev = this.tail;
            this.tail.next = item;
            this.tail = item;
        } else {
            this.head = this.tail = item;
        }
        ++this.fileCount;
    }

    public long getCacheSize() {
        return this.cacheSize;
    }

    private static class CacheItemComparator
    implements Comparator<FileItem> {
        private CacheItemComparator() {
        }

        @Override
        public int compare(FileItem o1, FileItem o2) {
            return (int)(o2.time - o1.time);
        }
    }

    private static class CacheFileComparator
    implements Comparator<File> {
        private CacheFileComparator() {
        }

        @Override
        public int compare(File o1, File o2) {
            return (int)(o2.lastModified() - o1.lastModified());
        }
    }

    private static class FileItem {
        final File file;
        final String name;
        final long time;
        final long size;
        final boolean isFile;

        private FileItem(File file) {
            this.file = file;
            this.time = file.lastModified();
            this.size = file.length();
            this.name = file.getName();
            this.isFile = file.isFile();
        }
    }

    private static class CacheItem {
        final String key;
        final File file;
        long size = 0L;
        long time = 0L;
        CacheItem prev = null;
        CacheItem next = null;

        CacheItem(String key, File file) {
            this.key = key;
            this.file = file;
        }
    }
}

