// NTLM (ntlm.js) authentication in JavaScript.
// The MIT License (MIT). Copyright (c) 2012 Erland Ranvinge.
define(['static.crypto.md4', 'static.crypto.des'], function(md4, des) {
var Msg = function(data) {
    this.data = [];
    if (!data) return;
    if (data.indexOf('NTLM ') === 0) data = data.substr(5);
    atob(data).split('').map(function(c) { this.push(c.charCodeAt(0)); }, this.data);
};

Msg.prototype.addByte = function(b) {
    this.data.push(b);
};

Msg.prototype.addShort = function(s) {
    this.data.push(s & 0xFF);
    this.data.push((s >> 8) & 0xFF);
};

Msg.prototype.addString = function(str, utf16) {
    if (utf16) // Fake UTF16 by padding each character in string.
        str = str.split('').map(function(c) { return (c + '\0'); }).join('');

    for (var i = 0; i < str.length; i++)
        this.data.push(str.charCodeAt(i));
};

Msg.prototype.getString = function(offset, length) {
    var result = '';
    for (var i = 0; i < length; i++) {
        if (offset + i >= this.data.length)
            return '';
        result += String.fromCharCode(this.data[offset + i]);
    }
    return result;
};

Msg.prototype.getByte = function(offset) {
    return this.data[offset];
};

var Ntlm = {};

Ntlm.error = function(msg) {
    console.error(msg);
};

Ntlm.createMessage1 = function(arg) {
    var hostname = arg.hostname, domain = arg.domain;
    var msg1 = new Msg();
    msg1.addString('NTLMSSP\0');
    msg1.addByte(1);
    msg1.addString('\0\0\0');
    msg1.addShort(0xb203);
    msg1.addString('\0\0');
    msg1.addShort(domain.length);
    msg1.addShort(domain.length);
    msg1.addShort(32 + hostname.length);
    msg1.addString('\0\0');
    msg1.addShort(hostname.length);
    msg1.addShort(hostname.length);
    msg1.addShort(32);
    msg1.addString('\0\0');
    msg1.addString(hostname.toUpperCase());
    msg1.addString(domain.toUpperCase());
    return msg1;
};

Ntlm.getChallenge = function(data) {
    var msg2 = new Msg(data);
    if (msg2.getString(0, 8) !== 'NTLMSSP\0') {
        Ntlm.error('Invalid NTLM response header.');
        return '';
    }
    if (msg2.getByte(8) !== 2) {
        Ntlm.error('Invalid NTLM response type.');
        return '';
    }
    var challenge = msg2.getString(24, 8);
    return challenge;
};

Ntlm.createMessage3 = function(challenge, arg) {
    var lmResponse = Ntlm.buildResponse(arg.lmHashedPassword, challenge);
    var ntResponse = Ntlm.buildResponse(arg.ntHashedPassword, challenge);
    var username = arg.username, hostname = arg.hostname, domain = arg.domain;
    var msg3 = new Msg();

    msg3.addString('NTLMSSP\0');
    msg3.addByte(3);
    msg3.addString('\0\0\0');

    msg3.addShort(24); // lmResponse
    msg3.addShort(24);
    msg3.addShort(64 + (domain.length + username.length + hostname.length) * 2);
    msg3.addString('\0\0');

    msg3.addShort(24); // ntResponse
    msg3.addShort(24);
    msg3.addShort(88 + (domain.length + username.length + hostname.length) * 2);
    msg3.addString('\0\0');

    msg3.addShort(domain.length * 2); // Domain.
    msg3.addShort(domain.length * 2);
    msg3.addShort(64);
    msg3.addString('\0\0');

    msg3.addShort(username.length * 2); // Username.
    msg3.addShort(username.length * 2);
    msg3.addShort(64 + domain.length * 2);
    msg3.addShort('\0\0');

    msg3.addShort(hostname.length * 2); // Hostname.
    msg3.addShort(hostname.length * 2);
    msg3.addShort(64 + (domain.length + username.length) * 2);
    msg3.addString('\0\0');

    msg3.addString('\0\0\0\0');
    msg3.addShort(112 + (domain.length + username.length + hostname.length) * 2);
    msg3.addString('\0\0');
    msg3.addShort(0x8201);
    msg3.addString('\0\0');

    msg3.addString(domain.toUpperCase(), true); // "Some" string are passed as UTF-16.
    msg3.addString(username, true);
    msg3.addString(hostname.toUpperCase(), true);
    msg3.addString(lmResponse);
    msg3.addString(ntResponse);

    return msg3;
};

Ntlm.createKey = function(str) {
    var key56 = [];
    while (str.length < 7) str += '\0';
    str = str.substr(0, 7);
    str.split('').map(function(c) { this.push(c.charCodeAt(0)); }, key56);
    var key = [0, 0, 0, 0, 0, 0, 0, 0];
    key[0] = key56[0]; // Convert 56 bit key to 64 bit.
    key[1] = ((key56[0] << 7) & 0xFF) | (key56[1] >> 1);
    key[2] = ((key56[1] << 6) & 0xFF) | (key56[2] >> 2);
    key[3] = ((key56[2] << 5) & 0xFF) | (key56[3] >> 3);
    key[4] = ((key56[3] << 4) & 0xFF) | (key56[4] >> 4);
    key[5] = ((key56[4] << 3) & 0xFF) | (key56[5] >> 5);
    key[6] = ((key56[5] << 2) & 0xFF) | (key56[6] >> 6);
    key[7] =  (key56[6] << 1) & 0xFF;
    for (var i = 0; i < key.length; i++) { // Fix DES key parity bits.
        var bit = 0;
        for (var k = 0; k < 7; k++) {
            t = key[i] >> k;
            bit = (t ^ bit) & 0x1;
        }
        key[i] = (key[i] & 0xFE) | bit;
    }

    var result = '';
    key.map(function(i) { result += String.fromCharCode(i); });
    return result;
};

Ntlm.buildResponse = function(key, text) {
    while (key.length < 21)
        key += '\0';
    var key1 = Ntlm.createKey(key.substr(0, 7));
    var key2 = Ntlm.createKey(key.substr(7, 7));
    var key3 = Ntlm.createKey(key.substr(14, 7));
    return des(key1, text, 1, 0) + des(key2, text, 1, 0) + des(key3, text, 1, 0);
};

Ntlm.createCredentials = function(arg) {
    var magic = 'KGS!@#$%'; // Create LM password hash.
    var password = arg.password || '';
    var lmPassword = password.toUpperCase().substr(0, 14);
    while (lmPassword.length < 14) lmPassword += '\0';
    var key1 = Ntlm.createKey(lmPassword);
    var key2 = Ntlm.createKey(lmPassword.substr(7));
    var lmHashedPassword = des(key1, magic, 1, 0) + des(key2, magic, 1, 0);

    var ntPassword = ''; // Create NT password hash.
    for (var i = 0; i < password.length; i++)
        ntPassword += password.charAt(i) + '\0';
    var ntHashedPassword = md4(ntPassword);

    return {
        domain: arg.domain || '',
        hostname: arg.hostname || '',
        username: arg.username,
        lmHashedPassword: lmHashedPassword,
        ntHashedPassword: ntHashedPassword
    };
};

Ntlm.isChallenge = function(xhr) {
    if (!xhr)
        return false;
    if (xhr.status !== 401)
        return false;
    var header = xhr.getResponseHeader('WWW-Authenticate');
    return header && header.indexOf('NTLM') !== -1;
};

return Ntlm;
});