﻿define(["require", "exports"], function (require, exports) {
    "use strict";
    var ByteArrayInputStream = (function () {
        function ByteArrayInputStream(data) {
            this.data = data;
            this.offs = 0;
            this.size = data.length;
        }
        ByteArrayInputStream.prototype.skip = function (size) {
            this.assertHas(size);
            this.offs += size;
        };
        ByteArrayInputStream.prototype.read = function () {
            var offs = this.offs;
            if (offs >= this.size)
                return -1;
            this.offs = offs + 1;
            return ByteArrayInputStream.byte(this.data, offs);
        };
        ByteArrayInputStream.prototype.readBytes = function (size) {
            this.assertHas(size);
            var offs = this.offs;
            this.offs = offs + size;
            return this.data.substr(offs, size);
        };
        ByteArrayInputStream.prototype.readU8 = function () {
            this.assertHas(1);
            var offs = this.offs;
            this.offs = offs + 1;
            return ByteArrayInputStream.byte(this.data, offs);
        };
        ByteArrayInputStream.prototype.readU32 = function () {
            return this.readI32(); //TODO
        };
        ByteArrayInputStream.prototype.readI32 = function () {
            this.assertHas(4);
            var data = this.data;
            var offs = this.offs;
            this.offs = offs + 4;
            var byte = ByteArrayInputStream.byte;
            return (byte(data, offs + 3) << 24) | (byte(data, offs + 2) << 16) | (byte(data, offs + 1) << 8) | byte(data, offs);
        };
        ;
        ByteArrayInputStream.prototype.assertHas = function (size) {
            var left = this.size - this.offs;
            if (left < size)
                throw Error('reader underflow: ' + left + ' < ' + size);
        };
        ByteArrayInputStream.byte = function (data, offs) {
            return data.charCodeAt(offs);
        };
        return ByteArrayInputStream;
    }());
    var TAG_EOF = 0;
    var TAG_LONG = 255;
    var TaggedReader = (function () {
        function TaggedReader(data) {
            this.ready = false;
            this.current_size = 0;
            this.current_tag = 0;
            this.stream = new ByteArrayInputStream(data);
        }
        TaggedReader.prototype.skip = function () {
            if (!this.ready)
                this.next();
            this.stream.skip(this.current_size);
            this.ready = false;
        };
        TaggedReader.prototype.next = function () {
            if (this.ready)
                this.skip();
            var stream = this.stream;
            var tag = stream.read();
            if (tag === -1) {
                this.current_size = 0;
                this.current_tag = TAG_EOF;
                return false;
            }
            if (tag === TAG_LONG) {
                tag = stream.read();
                this.current_size = stream.readU32();
            }
            else
                this.current_size = stream.readU8();
            this.current_tag = tag;
            this.ready = true;
            return true;
        };
        TaggedReader.prototype.getCurrentTag = function () {
            return this.current_tag;
        };
        TaggedReader.prototype.getInt = function () {
            this.assertReadySize(4);
            this.ready = false;
            return this.stream.readI32();
        };
        TaggedReader.prototype.getAnsi = function () {
            var cs = this.current_size;
            this.assertReadySize(cs);
            this.ready = false;
            return TaggedReader.bytestr_to_ansi(this.stream.readBytes(cs));
        };
        TaggedReader.prototype.assertReadySize = function (size) {
            if (!this.ready)
                throw Error('tag not ready');
            if (this.current_size !== size)
                throw Error('tag size mismatch ' + this.current_size + ' != ' + size);
        };
        TaggedReader.bytestr_to_ansi = function (bytestr) {
            function a2u(c) {
                if (c < 0x80)
                    return c;
                if (c === 0xA8)
                    return 1025;
                if (c === 0xB8)
                    return 1105;
                if (c >= 0xC0)
                    return c - 0xC0 + 1040;
                throw Error('?' + c);
            }
            var tmp = [];
            for (var i = 0, l = bytestr.length; i !== l; ++i)
                tmp.push(a2u(bytestr.charCodeAt(i)));
            return String.fromCharCode.apply(null, tmp);
        };
        return TaggedReader;
    }());
    exports.TaggedReader = TaggedReader;
});
